
 /****************************************************************************
 *   mainLi-ion.c                      
 *   Project: NXP LPC1111 Li-ion Charger
 *
 *   Description:
 *     This file contains the Li-ion Charger code for appnpote ANxxxx. The 
 *     code does not implement the temperature sensor test. 
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/

#include "LPC11xx.h"                        /* LPC11xx definitions */
#include "adc.h"
#include "timer16.h"
#include "gpio.h"


#define SYSTICK_DELAY	48000	/* 1 ms */ 
#define	PWMPERIOD 		250		/* the period of the timer when running at 192 kHz */


#define	VOLTS4p6   	1426	/* 0x592 */
#define	VOLTS4p2 	1302	/* 0x516 */
#define	VOLTS3	 	930		/* 0x3A2 */
#define	VOLT1  	 	310		/* 0x136 */

#define	MA350     	81		/* 0x051 */
#define	MA65	  	15		/* 0x00F */
#define	MA50	  	12		/* 0x007 */

#define	STARTDUTY   PWMPERIOD-50  	/* starting pwm duty cycle   */
#define	PWMDELTA    1  		  		/* may need to increase this */
//****************************************************************

#define	AD_VBATT    7    /* A/D channel# for bat voltage  */
#define	AD_IBATT    5    /* A/D channel# for bat current  */

#define	MIN50		3000000		/* 50 minutes in "TimeTick" (1 ms) counts	*/

/* ports pins for LEDs are sinking current , therefore logic inverted	*/
#define	GREENLEDON 	GPIOSetValue( 1, 1, 1 )	   /* Green LED on P1.1	*/
#define	GREENLEDOFF GPIOSetValue( 1, 1, 0 )
#define	REDLEDON 	GPIOSetValue( 1, 0, 1 )	   /* RED LED on P1.0   */
#define	REDLEDOFF 	GPIOSetValue( 1, 0, 0 )

#define	Blink_4096ms 0x0800;	/* LED blink rate depending upon the charging mode (4.1 seconds) */
#define	Blink_2048ms 0x0400;	/* LED blink rate depending upon the charging mode (2.0 seconds) */
#define	Blink_1024ms 0x0200;	/* LED blink rate depending upon the charging mode (1.0 seconds) */
#define	Blink_512ms  0x0100;	/* LED blink rate depending upon the charging mode (0.5 seconds) */

volatile uint32_t TimeTick = 0;
uint32_t LED_Mask, shadowPWM; 

int main(void);
void SysTick_Handler(void);
void precharge(void);
void CCcharge(void);
void CVcharge(void);
void timecharge(void);
void IOs_init(void);
void ChargeEnd(void);
void BatteryFaultCheck( int BatteryVoltage );


/*****************************************************************************
** Function name: main()		
**
** Descriptions: Main function for li-ion charger
**
** Three ADCs are used
**   AD5 measures battery voltage (plus the voltage across the current sense R)		
**	 AD6 measures battery temperature (not implemented)
**   AD7 measures the charge current by measuring the voltage across a 0.75 ohm resistor
**
** parameters:	none	
** Returned value:	none	
** 
*****************************************************************************/
int main() {

uint32_t VBAT;					/* VBAT is the voltage across the battery */
  
SystemInit();					/* initialize the system oscillator and PLL */

IOs_init();				  		/* initialize IOs for LEDs & PWM  */

ADCInit( ADC_CLK );		 		/* initialize the A/D converters used for this project */

SysTick_Config( SYSTICK_DELAY );	/* set systick for 1 msec interrupts */

/* check if the battery voltage is too low. The switch is off so this measures 
   the actual battery voltage without the PWM being on 	*/
VBAT = MeasureVBAT();
BatteryFaultCheck( VBAT );
																									   
/* Initialize the PWM in timer16_1 enabling match0 output   */
init_timer16PWM(1, PWMPERIOD, MATCH0, 0);
setMatch_timer16PWM (1, 0, 0); /* set output to on (pwm output = high)	*/	 
enable_timer16(1);			   /* start the pwm */

precharge();	/* start with the precharge stage if the battery voltage is <3V */
CCcharge();		/* charge at a constant 350mA */ 
CVcharge();		/* charge at a constant voltage until charge current is < 50mA */ 
timecharge();	/* charge at a constant voltage for 50 minutes (optional) */

GREENLEDON;		/* Green LED on indicates "Charging Complete"  */
ChargeEnd();  
}

/*****************************************************************************
** Function name: SysTick_Handler(void)		
**
** Descriptions:		
**
** parameters:	none	
** Returned value:	none	
** 
*****************************************************************************/
/* SysTick interrupt happens every 10 ms */
void SysTick_Handler( void ) {
  
TimeTick++;				/* increment tick counter used for timed charging */
  
if (TimeTick & LED_Mask)  /* toggle the Green LED  */
	GREENLEDOFF;
else
	GREENLEDON;
}

/*****************************************************************************
** Function name: precharge()		
**
** Descriptions: while VBATT < 3 volts, charge at 65mA 		
**
** parameters:	none	
** Returned value:	none	
** 
*****************************************************************************/

void  precharge( void ) {
	
uint32_t AD5, VBAT;						/* VBAT is the voltage across the battery */

LED_Mask = Blink_4096ms;				/* Blink Green LED every 4.1 sec */

do {	
	VBAT = MeasureVBAT();
	BatteryFaultCheck(VBAT);

	AD5 = MeasureCurrent();

	if ( AD5 < MA65 ) {					/* is current less than 65mA? */
		if (shadowPWM>0)		   		/* check to make sure the PWM does not underflow */
			shadowPWM -= PWMDELTA; 		/* increase PWM duty cycle */
		}
	else {
		if( AD5 > MA65 ) {				/* is current greater than 65mA? */
			if (shadowPWM < PWMPERIOD)	/* check to make sure the PWM does not overflow */
				shadowPWM += PWMDELTA;	/* decrease PWM duty cycle */
				}
		}
	setMatch_timer16PWM (1, 0, shadowPWM );				

	__wfi();							/* go to sleep until next Systick Interrupt */
   } while ( VBAT < VOLTS3 ); 			/* precharge continues while VBAT < 3V      */ 
}

/*****************************************************************************
** Function name: CCcharge()		
**
** Descriptions: charge at a constant 350mA		
**
** parameters:	none	
** Returned value:	none	
** 
*****************************************************************************/

void  CCcharge( void ) {

uint32_t AD5, VBAT;
uint32_t NoiseFilter = 0;	/* NoiseFilter is a counter to ensure the cycle doesn't
							   complete too soon due to a noise event or ripple voltage on VBAT.
							   It must exceed the 4.2V for more than 10 consecutive measurements */	
	
LED_Mask = Blink_2048ms;	/* Blink the Green LED every 2.05 seconds */

while (NoiseFilter < 10) {
	
	do {
		VBAT = MeasureVBAT();
		BatteryFaultCheck(VBAT);	/* check if VBAT is too high indicating battery removal */
		
		if (VBAT < VOLTS4p2) NoiseFilter = 0;	/* Reset the "noise filter" if the VBAT<4.2V */ 			

		AD5 = MeasureCurrent();	 	/* measure the battery current */

		if ( AD5 < MA350 ) {		/* is current less than 350mA */
			if (shadowPWM>0)		/* check to make sure the PWM does not underflow */
				shadowPWM -= PWMDELTA;		/* increase PWM duty cycle */
			}
		else {
		if ( AD5 > MA350 ) {			/* is current greater than 350mA */
			if (shadowPWM < PWMPERIOD)	/* check to make sure the PWM does not overflow */
				shadowPWM += PWMDELTA;	/* decrease PWM duty cycle */
			}
		}	 
		setMatch_timer16PWM (1, 0, shadowPWM ); /*  update the PWM  */

		__wfi();					 /* put in sleep mode, waiting for Systick interrupt */
											 
		} while ( VBAT < VOLTS4p2 ); /* constant current charging while battery voltage less than 4.2V */
	
	NoiseFilter++;
	}
}

/*****************************************************************************
** Function name: CVcharge()		
**
** Descriptions: while IBATT > 50mA, charge at a contstant 4.20 volts	
**
** parameters:		
** Returned value:		
** 
*****************************************************************************/
void  CVcharge( void ) { 
 
uint32_t AD5, VBAT;

uint32_t NoiseFilter = 0;	/* NoiseFilter is a counter to ensure the cycle doesn't
							   complete too soon due to a noise event or ripple voltage.
							   The current must be less than 50mA for 10 consecutive measurements */	


LED_Mask = Blink_1024ms;		/* Blink the Green LED every 1 seconds */

while (NoiseFilter < 10) {

	do {																			   
		VBAT = MeasureVBAT();		/*  obtain the battery voltage */
		BatteryFaultCheck(VBAT);	/* check if VBAT is too high indicating battery removal */
							 
		if ( VBAT < VOLTS4p2 ) {	/*  is battery voltage less than 4.2V?  */
			if (shadowPWM>0)		/*  check to make sure the PWM does not underflow */
				shadowPWM -= PWMDELTA;	/*  increase duty cycle 		*/
	   		}
		else {
		if ( VBAT > VOLTS4p2 ) {	/*  is battery voltage greater than 4.2V?  */
			if (shadowPWM < PWMPERIOD)		/*  check to make sure the PWM does not overflow */
				shadowPWM += PWMDELTA;	/*  decrease duty cycle 		*/
	     	}
		}

	setMatch_timer16PWM (1, 0, shadowPWM );	/*  update the PWM 				*/

	__wfi();					/* put in sleep mode, waiting for Systick interrupt */

	AD5 = MeasureCurrent();		/* measure voltage across current sense resistor */

	if ( AD5 > MA50) NoiseFilter = 0;	/* reset the counter if current is less than 50mA */

	} while ( AD5 > MA50 );  	/* continue to charge while the battery charge current > 50mA */

	NoiseFilter++;
	}

}

/*****************************************************************************
** Function name: timecharge()		
**
** Descriptions: charge at a constant 4.2 volts for 50 minutes		
**
** parameters:	none	
** Returned value:	none	
** 
*****************************************************************************/
void   timecharge( void )	{

uint32_t VBAT;

TimeTick = 0;		/* reset the time counter */

LED_Mask = Blink_512ms;			/* Blink the Green LED every 0.512 ms */

while (TimeTick < MIN50) {		

	VBAT = MeasureVBAT();
	BatteryFaultCheck( VBAT );	/* check if battery is still connected */

	if ( VBAT < VOLTS4p2 ) {
		if (shadowPWM>0)			/*  check to make sure the PWM does not underflow */
			shadowPWM -= PWMDELTA;	/*  increase duty cycle */
	 }
    else {
		if ( VBAT > VOLTS4p2 ) {
		if (shadowPWM < PWMPERIOD)	/*  check to make sure the PWM does not overflow */
			shadowPWM += PWMDELTA;	/*  decrease duty cycle */
	 	}
	}
	setMatch_timer16PWM (1, 0, shadowPWM );	/*  update the PWM */

	__wfi();
	}
}

/*****************************************************************************
** Function name: IOs_init()		
**
** Descriptions: configure IOs for Li-ion charger		
**
** parameters:		
** Returned value:		
** 
*****************************************************************************/
void  IOs_init() {
	 
LPC_SYSCON->SYSAHBCLKCTRL &= ~(1<<11); /* turn off power to SSP0 block since we are not using it */

LPC_IOCON->PIO1_9  &= ~0x1F;	   	/* clear all options for PWM pin 			*/
LPC_IOCON->R_PIO1_0 = 0x000000C1;	/* Configure P1.0 as output with no pull-up */
LPC_IOCON->R_PIO1_1 = 0x000000C1;	/* Configure P1.1 as output with no pull-up */
GREENLEDOFF;			  			/* start with LEDs off						*/
REDLEDOFF;
GPIOSetDir(1, 0, 1);   				/*  P1.0 (RED LED) is an output 	*/
GPIOSetDir(1, 1, 1);	  			/*  P1.1 (GREEN LED) is an output 	*/

}


/*****************************************************************************
** Function name: ChargeEnd()		
**
** Descriptions:  Stops the systick timer, turns off the PWM		
**
** parameters:		
** Returned value:		
** 
*****************************************************************************/
void ChargeEnd( void ) {
SysTick->CTRL = (1 << SYSTICK_CLKSOURCE) | (0<<SYSTICK_ENABLE) | (0<<SYSTICK_TICKINT);	/* turn off Systick interrupts */
LPC_IOCON->PIO1_9 &= ~0x07;	   	/* turn off PWM			   		*/
LPC_IOCON->PIO1_9  = (1<<3);	/* set pulldown on PWM output	*/

__wfi(); /* put in low power state. there should be no interrupts at this point since Systick is disabled */

}

/*****************************************************************************
** Function name: BatteryRemovalCheck()		
**
** Descriptions:  Turns on Red LED, turns off green LED, and jumps to ChargeEnd()		
**
** parameters:	takes the battery voltage	
** Returned value:		
** 
*****************************************************************************/
void BatteryFaultCheck( int BatteryVoltage ) {
   
if ( (BatteryVoltage > VOLTS4p6) || (BatteryVoltage  < VOLT1) ) { 	/* if battery removed or shorted, turn off charger */
	REDLEDON;
	GREENLEDOFF;
	ChargeEnd();
	}
}
